/**
 * upload-ui.js - Upload tab UI logic
 */

import { processUploadedFile } from './memory-extractor.js';

let userId = null;
let apiKey = null;

export function init(currentUserId, currentApiKey) {
    userId = currentUserId;
    apiKey = currentApiKey;

    setupUploadListeners();
}

function setupUploadListeners() {
    const fileInput = document.getElementById('file-upload');
    const uploadBtn = document.getElementById('upload-btn');
    const dropZone = document.getElementById('drop-zone');

    if (uploadBtn && fileInput) {
        uploadBtn.addEventListener('click', () => fileInput.click());
    }

    if (fileInput) {
        fileInput.addEventListener('change', handleFileSelect);
    }

    if (dropZone) {
        dropZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            dropZone.classList.add('drag-over');
        });

        dropZone.addEventListener('dragleave', () => {
            dropZone.classList.remove('drag-over');
        });

        dropZone.addEventListener('drop', (e) => {
            e.preventDefault();
            dropZone.classList.remove('drag-over');
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                processFile(files[0]);
            }
        });
    }
}

function handleFileSelect(e) {
    const files = e.target.files;
    if (files.length > 0) {
        processFile(files[0]);
    }
}

async function processFile(file) {
    if (!apiKey) {
        alert('Please add your OpenAI API key in Settings first!');
        return;
    }

    const progressEl = document.getElementById('upload-progress');
    const statusEl = document.getElementById('upload-status');

    try {
        if (progressEl) progressEl.style.display = 'block';

        const result = await processUploadedFile(apiKey, file, userId, (msg) => {
            if (statusEl) statusEl.textContent = msg;
        });

        if (statusEl) {
            statusEl.textContent = `✅ Complete! ${result.memoryCount} memories extracted.`;
        }

        setTimeout(() => {
            if (progressEl) progressEl.style.display = 'none';
        }, 3000);

    } catch (error) {
        if (statusEl) statusEl.textContent = `❌ Error: ${error.message}`;
    }
}
