/**
 * storage.js - IndexedDB wrapper for local data storage
 *
 * This module handles all data persistence for the app using IndexedDB.
 * Students will learn how to store structured data in the browser without a backend.
 *
 * Object Stores:
 * 1. users - User accounts (id, username, email, passwordHash, createdAt)
 * 2. conversations - ChatGPT conversation data (id, userId, title, messages, createdAt)
 * 3. memories - Extracted memories (id, userId, type, title, content, conversationId, createdAt)
 */

const DB_NAME = 'ContextAILocal';
const DB_VERSION = 1;

// Database instance (shared across functions)
let db = null;

/**
 * Initialize the IndexedDB database
 * Creates object stores and indexes if they don't exist
 */
export async function initDatabase() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open(DB_NAME, DB_VERSION);

        // Handle database upgrade (first time or version change)
        request.onupgradeneeded = (event) => {
            const database = event.target.result;

            // Create 'users' object store
            if (!database.objectStoreNames.contains('users')) {
                const usersStore = database.createObjectStore('users', {
                    keyPath: 'id',
                    autoIncrement: true
                });
                // Index for fast lookup by email (for login)
                usersStore.createIndex('email', 'email', { unique: true });
                console.log('✅ Created users object store');
            }

            // Create 'conversations' object store
            if (!database.objectStoreNames.contains('conversations')) {
                const convsStore = database.createObjectStore('conversations', {
                    keyPath: 'id',
                    autoIncrement: true
                });
                // Index for filtering conversations by user
                convsStore.createIndex('userId', 'userId', { unique: false });
                console.log('✅ Created conversations object store');
            }

            // Create 'memories' object store
            if (!database.objectStoreNames.contains('memories')) {
                const memsStore = database.createObjectStore('memories', {
                    keyPath: 'id',
                    autoIncrement: true
                });
                // Indexes for filtering and searching memories
                memsStore.createIndex('userId', 'userId', { unique: false });
                memsStore.createIndex('type', 'type', { unique: false });
                memsStore.createIndex('userId_type', ['userId', 'type'], { unique: false });
                console.log('✅ Created memories object store');
            }
        };

        request.onsuccess = (event) => {
            db = event.target.result;
            console.log('✅ Database initialized successfully');
            resolve(db);
        };

        request.onerror = (event) => {
            console.error('❌ Database initialization failed:', event.target.error);
            reject(event.target.error);
        };
    });
}

// ============================================================================
// USER OPERATIONS
// ============================================================================

/**
 * Create a new user account
 * @param {Object} user - User data { username, email, passwordHash }
 * @returns {Promise<number>} User ID
 */
export async function createUser(user) {
    const transaction = db.transaction(['users'], 'readwrite');
    const store = transaction.objectStore('users');

    const userData = {
        ...user,
        createdAt: new Date().toISOString()
    };

    return new Promise((resolve, reject) => {
        const request = store.add(userData);
        request.onsuccess = () => resolve(request.result);
        request.onerror = () => reject(request.error);
    });
}

/**
 * Find user by email (for login)
 * @param {string} email
 * @returns {Promise<Object|null>} User object or null
 */
export async function getUserByEmail(email) {
    const transaction = db.transaction(['users'], 'readonly');
    const store = transaction.objectStore('users');
    const index = store.index('email');

    return new Promise((resolve, reject) => {
        const request = index.get(email);
        request.onsuccess = () => resolve(request.result || null);
        request.onerror = () => reject(request.error);
    });
}

/**
 * Get user by ID
 * @param {number} userId
 * @returns {Promise<Object|null>}
 */
export async function getUserById(userId) {
    const transaction = db.transaction(['users'], 'readonly');
    const store = transaction.objectStore('users');

    return new Promise((resolve, reject) => {
        const request = store.get(userId);
        request.onsuccess = () => resolve(request.result || null);
        request.onerror = () => reject(request.error);
    });
}

// ============================================================================
// CONVERSATION OPERATIONS
// ============================================================================

/**
 * Save a conversation to the database
 * @param {Object} conversation - { userId, title, messages, createdAt }
 * @returns {Promise<number>} Conversation ID
 */
export async function saveConversation(conversation) {
    const transaction = db.transaction(['conversations'], 'readwrite');
    const store = transaction.objectStore('conversations');

    const convData = {
        ...conversation,
        uploadedAt: new Date().toISOString()
    };

    return new Promise((resolve, reject) => {
        const request = store.add(convData);
        request.onsuccess = () => resolve(request.result);
        request.onerror = () => reject(request.error);
    });
}

/**
 * Get all conversations for a user
 * @param {number} userId
 * @returns {Promise<Array>} Array of conversations
 */
export async function getConversations(userId) {
    const transaction = db.transaction(['conversations'], 'readonly');
    const store = transaction.objectStore('conversations');
    const index = store.index('userId');

    return new Promise((resolve, reject) => {
        const request = index.getAll(userId);
        request.onsuccess = () => resolve(request.result || []);
        request.onerror = () => reject(request.error);
    });
}

/**
 * Get a single conversation by ID
 * @param {number} conversationId
 * @returns {Promise<Object|null>}
 */
export async function getConversationById(conversationId) {
    const transaction = db.transaction(['conversations'], 'readonly');
    const store = transaction.objectStore('conversations');

    return new Promise((resolve, reject) => {
        const request = store.get(conversationId);
        request.onsuccess = () => resolve(request.result || null);
        request.onerror = () => reject(request.error);
    });
}

// ============================================================================
// MEMORY OPERATIONS
// ============================================================================

/**
 * Save a memory to the database
 * @param {Object} memory - { userId, type, title, content, conversationId }
 * @returns {Promise<number>} Memory ID
 */
export async function saveMemory(memory) {
    const transaction = db.transaction(['memories'], 'readwrite');
    const store = transaction.objectStore('memories');

    const memoryData = {
        ...memory,
        createdAt: new Date().toISOString()
    };

    return new Promise((resolve, reject) => {
        const request = store.add(memoryData);
        request.onsuccess = () => resolve(request.result);
        request.onerror = () => reject(request.error);
    });
}

/**
 * Get all memories for a user
 * @param {number} userId
 * @param {string} [type] - Optional filter by type (episodic, entity, fact)
 * @returns {Promise<Array>} Array of memories
 */
export async function getMemories(userId, type = null) {
    const transaction = db.transaction(['memories'], 'readonly');
    const store = transaction.objectStore('memories');

    return new Promise((resolve, reject) => {
        let request;

        if (type) {
            // Filter by userId AND type
            const index = store.index('userId_type');
            request = index.getAll([userId, type]);
        } else {
            // Get all memories for user
            const index = store.index('userId');
            request = index.getAll(userId);
        }

        request.onsuccess = () => {
            const memories = request.result || [];
            // Sort by newest first
            memories.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
            resolve(memories);
        };
        request.onerror = () => reject(request.error);
    });
}

/**
 * Search memories by keyword (simple text search)
 * Students will learn: This is a basic search. For production, you'd use embeddings!
 *
 * @param {number} userId
 * @param {string} keyword
 * @param {string} [type] - Optional type filter
 * @returns {Promise<Array>} Matching memories
 */
export async function searchMemories(userId, keyword, type = null) {
    // First get all memories (optionally filtered by type)
    const memories = await getMemories(userId, type);

    // Simple keyword search (case-insensitive)
    const searchTerm = keyword.toLowerCase();

    return memories.filter(memory => {
        const titleMatch = memory.title.toLowerCase().includes(searchTerm);
        const contentMatch = memory.content.toLowerCase().includes(searchTerm);
        return titleMatch || contentMatch;
    });
}

/**
 * Get memory count by type
 * Useful for displaying stats in the UI
 *
 * @param {number} userId
 * @returns {Promise<Object>} { episodic: 10, entity: 5, fact: 8, total: 23 }
 */
export async function getMemoryStats(userId) {
    const memories = await getMemories(userId);

    const stats = {
        episodic: 0,
        entity: 0,
        fact: 0,
        total: memories.length
    };

    memories.forEach(memory => {
        if (memory.type in stats) {
            stats[memory.type]++;
        }
    });

    return stats;
}

/**
 * Delete a memory by ID
 * @param {number} memoryId
 * @returns {Promise<void>}
 */
export async function deleteMemory(memoryId) {
    const transaction = db.transaction(['memories'], 'readwrite');
    const store = transaction.objectStore('memories');

    return new Promise((resolve, reject) => {
        const request = store.delete(memoryId);
        request.onsuccess = () => resolve();
        request.onerror = () => reject(request.error);
    });
}

/**
 * Delete all memories for a user
 * @param {number} userId
 * @returns {Promise<number>} Number of deleted memories
 */
export async function deleteAllMemories(userId) {
    const memories = await getMemories(userId);
    const transaction = db.transaction(['memories'], 'readwrite');
    const store = transaction.objectStore('memories');

    let deleteCount = 0;

    return new Promise((resolve, reject) => {
        memories.forEach((memory, index) => {
            const request = store.delete(memory.id);
            request.onsuccess = () => {
                deleteCount++;
                // Resolve when all deletes are done
                if (index === memories.length - 1) {
                    resolve(deleteCount);
                }
            };
            request.onerror = () => reject(request.error);
        });

        // Handle edge case of zero memories
        if (memories.length === 0) {
            resolve(0);
        }
    });
}

// ============================================================================
// UTILITY FUNCTIONS
// ============================================================================

/**
 * Clear all data from the database (for testing or reset)
 * @returns {Promise<void>}
 */
export async function clearAllData() {
    const transaction = db.transaction(['users', 'conversations', 'memories'], 'readwrite');

    const promises = [
        new Promise((resolve, reject) => {
            const request = transaction.objectStore('users').clear();
            request.onsuccess = () => resolve();
            request.onerror = () => reject(request.error);
        }),
        new Promise((resolve, reject) => {
            const request = transaction.objectStore('conversations').clear();
            request.onsuccess = () => resolve();
            request.onerror = () => reject(request.error);
        }),
        new Promise((resolve, reject) => {
            const request = transaction.objectStore('memories').clear();
            request.onsuccess = () => resolve();
            request.onerror = () => reject(request.error);
        })
    ];

    await Promise.all(promises);
    console.log('✅ All data cleared');
}

/**
 * Export all user data as JSON (for backup or portability)
 * @param {number} userId
 * @returns {Promise<Object>} { user, conversations, memories }
 */
export async function exportUserData(userId) {
    const user = await getUserById(userId);
    const conversations = await getConversations(userId);
    const memories = await getMemories(userId);

    return {
        user: { ...user, passwordHash: undefined }, // Don't export password
        conversations,
        memories,
        exportedAt: new Date().toISOString()
    };
}

// Initialize database when module loads
console.log('📦 Storage module loaded');
