/**
 * memory-ui.js - Memories tab UI logic
 */

import { getMemories, searchMemories, getMemoryStats } from './storage.js';

let userId = null;
let currentFilter = 'all';

export async function init(currentUserId) {
    userId = currentUserId;
    await loadMemories();
    setupMemoryListeners();
}

function setupMemoryListeners() {
    // Filter buttons
    document.querySelectorAll('.filter-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            currentFilter = btn.getAttribute('data-filter');
            document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
            btn.classList.add('active');
            loadMemories();
        });
    });

    // Search input
    const searchInput = document.getElementById('memory-search');
    if (searchInput) {
        searchInput.addEventListener('input', (e) => {
            const keyword = e.target.value.trim();
            if (keyword) {
                searchMemoriesAndDisplay(keyword);
            } else {
                loadMemories();
            }
        });
    }
}

async function loadMemories() {
    const filterType = currentFilter === 'all' ? null : currentFilter;
    const memories = await getMemories(userId, filterType);
    displayMemories(memories);
}

async function searchMemoriesAndDisplay(keyword) {
    const filterType = currentFilter === 'all' ? null : currentFilter;
    const memories = await searchMemories(userId, keyword, filterType);
    displayMemories(memories);
}

function displayMemories(memories) {
    const container = document.getElementById('memories-grid');
    if (!container) return;

    if (memories.length === 0) {
        container.innerHTML = '<p class="empty-state">No memories found. Upload conversations to get started!</p>';
        return;
    }

    container.innerHTML = memories.map(memory => `
        <div class="memory-card" data-type="${memory.type}">
            <div class="memory-type-badge">${memory.type}</div>
            <h3 class="memory-title">${escapeHtml(memory.title)}</h3>
            <p class="memory-content">${escapeHtml(memory.content)}</p>
            <div class="memory-date">${formatDate(memory.createdAt)}</div>
        </div>
    `).join('');
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString();
}
