/**
 * memory-extractor.js - OpenAI-powered memory extraction
 *
 * This is THE CORE PATTERN of Memory-First Architecture!
 *
 * Students will learn:
 * - How to use OpenAI API to extract structured data from conversations
 * - The difference between episodic memories, entities, and facts
 * - How to design prompts that return consistent JSON
 * - How AI transforms raw conversations into persistent context
 *
 * This is the "magic" that makes AI apps sticky - turning one-time interactions
 * into a growing knowledge base that improves over time.
 */

import { saveConversation, saveMemory } from './storage.js';

// OpenAI API configuration
const OPENAI_API_URL = 'https://api.openai.com/v1/chat/completions';
const DEFAULT_MODEL = 'gpt-4o-mini'; // Cost-effective model for extraction

/**
 * THE MEMORY EXTRACTION PROMPT
 *
 * This is the key to Memory-First Architecture!
 * Students can modify this prompt to extract different types of information.
 *
 * Important: We ask for JSON output to get structured, parseable data.
 */
const MEMORY_EXTRACTION_PROMPT = `You are a memory extraction AI. Your job is to analyze a conversation and extract three types of memories:

1. EPISODIC MEMORIES: Specific events, experiences, or moments in time
   - Example: "Discussed building a personal AI database on Oct 10, 2024"
   - Example: "Had trouble with React state management in the morning"

2. ENTITIES: People, places, organizations, or important concepts mentioned
   - Example: "OpenAI - AI research company that created ChatGPT"
   - Example: "Nathan - AI educator building 12 products in 12 weeks"

3. FACTS: User preferences, skills, knowledge, opinions, or learned information
   - Example: "Prefers vanilla JavaScript over React for teaching"
   - Example: "Knows how to use IndexedDB for browser storage"

INSTRUCTIONS:
- Extract 3-10 memories per category (or fewer if the conversation is short)
- Each memory should have a concise title (max 60 chars) and detailed content (max 200 chars)
- Focus on information that would be useful for personalizing future AI interactions
- Avoid extracting generic or obvious information

Return your response as valid JSON in this EXACT format:
{
  "episodic": [
    {"title": "Event title", "content": "Detailed description of what happened"}
  ],
  "entities": [
    {"title": "Entity name", "content": "Description of the entity and its relevance"}
  ],
  "facts": [
    {"title": "Fact title", "content": "Detailed fact about the user"}
  ]
}`;

/**
 * Extract memories from a single conversation using OpenAI
 *
 * @param {string} apiKey - User's OpenAI API key
 * @param {Object} conversation - ChatGPT conversation object
 * @param {Function} onProgress - Optional callback for progress updates
 * @returns {Promise<Object>} { episodic: [], entities: [], facts: [] }
 */
export async function extractMemoriesFromConversation(apiKey, conversation, onProgress = null) {
    try {
        // Build conversation text from messages
        const conversationText = buildConversationText(conversation);

        if (onProgress) {
            onProgress(`Analyzing conversation: "${conversation.title || 'Untitled'}"...`);
        }

        // Call OpenAI API
        const response = await fetch(OPENAI_API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            },
            body: JSON.stringify({
                model: DEFAULT_MODEL,
                messages: [
                    {
                        role: 'system',
                        content: MEMORY_EXTRACTION_PROMPT
                    },
                    {
                        role: 'user',
                        content: `Analyze this conversation and extract memories:\n\n${conversationText}`
                    }
                ],
                temperature: 0.3, // Lower temperature for more consistent output
                max_tokens: 2000
            })
        });

        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error?.message || 'OpenAI API request failed');
        }

        const data = await response.json();
        const content = data.choices[0].message.content;

        // Parse JSON response
        const memories = parseMemoryResponse(content);

        if (onProgress) {
            const total = memories.episodic.length + memories.entities.length + memories.facts.length;
            onProgress(`✅ Extracted ${total} memories from "${conversation.title || 'Untitled'}"`);
        }

        return memories;
    } catch (error) {
        console.error('Memory extraction error:', error);
        if (onProgress) {
            onProgress(`❌ Failed to extract memories: ${error.message}`);
        }
        throw error;
    }
}

/**
 * Process an uploaded ChatGPT export file and extract all memories
 *
 * This is the main function students will call from the UI!
 *
 * @param {string} apiKey - OpenAI API key
 * @param {File} file - Uploaded JSON file
 * @param {number} userId - Current user's ID
 * @param {Function} onProgress - Progress callback
 * @returns {Promise<Object>} { conversationCount, memoryCount, conversations, memories }
 */
export async function processUploadedFile(apiKey, file, userId, onProgress = null) {
    try {
        // Validate API key
        if (!apiKey || apiKey.trim().length === 0) {
            throw new Error('OpenAI API key is required. Please add it in Settings.');
        }

        if (onProgress) {
            onProgress('📖 Reading uploaded file...');
        }

        // Read file content
        const fileContent = await readFileAsText(file);
        const data = JSON.parse(fileContent);

        // ChatGPT exports are an array of conversation objects
        if (!Array.isArray(data)) {
            throw new Error('Invalid ChatGPT export format. Expected an array of conversations.');
        }

        if (onProgress) {
            onProgress(`✅ Found ${data.length} conversations in file`);
        }

        // Process each conversation
        const results = {
            conversationCount: 0,
            memoryCount: 0,
            conversations: [],
            memories: []
        };

        for (let i = 0; i < data.length; i++) {
            const conversation = data[i];

            if (onProgress) {
                onProgress(`Processing conversation ${i + 1}/${data.length}...`);
            }

            try {
                // Save conversation to database
                const conversationId = await saveConversation({
                    userId,
                    title: conversation.title || `Conversation ${i + 1}`,
                    messages: extractMessages(conversation),
                    createdAt: new Date(conversation.create_time * 1000).toISOString()
                });

                results.conversationCount++;
                results.conversations.push({ id: conversationId, title: conversation.title });

                // Extract memories using OpenAI
                const extractedMemories = await extractMemoriesFromConversation(
                    apiKey,
                    conversation,
                    onProgress
                );

                // Save memories to database
                await saveExtractedMemories(userId, conversationId, extractedMemories);

                const memoryCount = extractedMemories.episodic.length +
                                   extractedMemories.entities.length +
                                   extractedMemories.facts.length;

                results.memoryCount += memoryCount;
                results.memories.push(...extractedMemories.episodic, ...extractedMemories.entities, ...extractedMemories.facts);

            } catch (convError) {
                console.error(`Error processing conversation ${i + 1}:`, convError);
                if (onProgress) {
                    onProgress(`⚠️ Skipped conversation ${i + 1}: ${convError.message}`);
                }
            }

            // Small delay to avoid rate limiting
            await sleep(500);
        }

        if (onProgress) {
            onProgress(`🎉 Complete! Processed ${results.conversationCount} conversations, extracted ${results.memoryCount} memories.`);
        }

        return results;
    } catch (error) {
        console.error('File processing error:', error);
        throw error;
    }
}

// ============================================================================
// HELPER FUNCTIONS
// ============================================================================

/**
 * Build readable conversation text from ChatGPT conversation object
 */
function buildConversationText(conversation) {
    const messages = extractMessages(conversation);

    return messages.map(msg => {
        const role = msg.role === 'user' ? 'User' : 'Assistant';
        return `${role}: ${msg.content}`;
    }).join('\n\n');
}

/**
 * Extract messages array from ChatGPT conversation structure
 * ChatGPT exports have a complex nested structure - this simplifies it
 */
function extractMessages(conversation) {
    const messages = [];

    if (!conversation.mapping) {
        return messages;
    }

    // ChatGPT's mapping structure is a bit complex - we need to traverse it
    const mapping = conversation.mapping;

    // Build message tree
    for (const nodeId in mapping) {
        const node = mapping[nodeId];
        if (node.message && node.message.content && node.message.content.parts) {
            const content = node.message.content.parts.join(' ');
            const role = node.message.author.role;

            if (content.trim() && (role === 'user' || role === 'assistant')) {
                messages.push({
                    role: role,
                    content: content.trim()
                });
            }
        }
    }

    return messages;
}

/**
 * Parse OpenAI's JSON response
 * Handles cases where the model wraps JSON in markdown code blocks
 */
function parseMemoryResponse(content) {
    try {
        // Remove markdown code blocks if present
        let jsonText = content.trim();

        if (jsonText.startsWith('```')) {
            jsonText = jsonText.replace(/```json\n?/g, '').replace(/```\n?/g, '');
        }

        const parsed = JSON.parse(jsonText);

        // Ensure all arrays exist
        return {
            episodic: parsed.episodic || [],
            entities: parsed.entities || [],
            facts: parsed.facts || []
        };
    } catch (error) {
        console.error('Failed to parse memory response:', content);
        throw new Error('AI returned invalid JSON. Please try again.');
    }
}

/**
 * Save extracted memories to database
 */
async function saveExtractedMemories(userId, conversationId, memories) {
    // Save episodic memories
    for (const memory of memories.episodic) {
        await saveMemory({
            userId,
            conversationId,
            type: 'episodic',
            title: memory.title,
            content: memory.content
        });
    }

    // Save entities
    for (const memory of memories.entities) {
        await saveMemory({
            userId,
            conversationId,
            type: 'entity',
            title: memory.title,
            content: memory.content
        });
    }

    // Save facts
    for (const memory of memories.facts) {
        await saveMemory({
            userId,
            conversationId,
            type: 'fact',
            title: memory.title,
            content: memory.content
        });
    }
}

/**
 * Read file as text (Promise-based)
 */
function readFileAsText(file) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = (e) => resolve(e.target.result);
        reader.onerror = (e) => reject(e);
        reader.readAsText(file);
    });
}

/**
 * Sleep helper for rate limiting
 */
function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

/**
 * Test API key validity
 * @param {string} apiKey
 * @returns {Promise<boolean>}
 */
export async function testApiKey(apiKey) {
    try {
        const response = await fetch(OPENAI_API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            },
            body: JSON.stringify({
                model: DEFAULT_MODEL,
                messages: [{ role: 'user', content: 'Hi' }],
                max_tokens: 5
            })
        });

        return response.ok;
    } catch (error) {
        return false;
    }
}

console.log('🧠 Memory extraction module loaded');
