/**
 * chat.js - Chat interface with memory context
 *
 * This module demonstrates Memory-First Architecture in action!
 *
 * Students will learn:
 * - How to inject memory context into AI conversations
 * - How to use OpenAI's streaming API for real-time responses
 * - The difference between AI with and without memory
 * - How context transforms a chatbot into a personal assistant
 *
 * Key insight: By including ALL memories in the system prompt, the AI "knows"
 * the user and can provide personalized, contextual responses.
 */

import { getMemories } from './storage.js';

const OPENAI_API_URL = 'https://api.openai.com/v1/chat/completions';
const DEFAULT_MODEL = 'gpt-4o-mini';

// Chat history (in-memory, resets on page refresh)
let chatHistory = [];

/**
 * Send a message and get AI response with memory context
 *
 * This is THE KEY FUNCTION that shows Memory-First Architecture!
 *
 * @param {string} apiKey - OpenAI API key
 * @param {number} userId - Current user ID
 * @param {string} userMessage - User's message
 * @param {Function} onChunk - Callback for streaming response chunks
 * @param {Function} onComplete - Callback when response is complete
 * @returns {Promise<string>} Full AI response
 */
export async function sendMessage(apiKey, userId, userMessage, onChunk = null, onComplete = null) {
    try {
        // Get all user's memories
        const memories = await getMemories(userId);

        // Build system prompt with memory context
        const systemPrompt = buildSystemPromptWithMemories(memories);

        // Add user message to chat history
        chatHistory.push({
            role: 'user',
            content: userMessage
        });

        // Build messages array for API
        const messages = [
            { role: 'system', content: systemPrompt },
            ...chatHistory
        ];

        // Call OpenAI API with streaming
        const response = await fetch(OPENAI_API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${apiKey}`
            },
            body: JSON.stringify({
                model: DEFAULT_MODEL,
                messages: messages,
                stream: true, // Enable streaming!
                temperature: 0.7,
                max_tokens: 1000
            })
        });

        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error?.message || 'API request failed');
        }

        // Process streaming response
        const fullResponse = await processStreamingResponse(response, onChunk);

        // Add assistant response to chat history
        chatHistory.push({
            role: 'assistant',
            content: fullResponse
        });

        if (onComplete) {
            onComplete(fullResponse);
        }

        return fullResponse;
    } catch (error) {
        console.error('Chat error:', error);
        throw error;
    }
}

/**
 * Build system prompt with ALL user memories
 *
 * Students: This is where the magic happens!
 * We're giving the AI FULL CONTEXT about the user.
 *
 * Note: In production, you'd want to:
 * 1. Limit to most relevant memories (semantic search)
 * 2. Stay within token limits
 * 3. Prioritize recent/important memories
 *
 * But for teaching, showing ALL memories is simpler and more transparent.
 */
function buildSystemPromptWithMemories(memories) {
    let prompt = `You are a helpful personal AI assistant with access to the user's memories.

Your job is to provide personalized, contextual responses based on what you know about the user.

`;

    if (memories.length === 0) {
        prompt += `The user has no memories stored yet. Respond helpfully based only on the current conversation.`;
        return prompt;
    }

    // Group memories by type
    const episodic = memories.filter(m => m.type === 'episodic');
    const entities = memories.filter(m => m.type === 'entity');
    const facts = memories.filter(m => m.type === 'fact');

    prompt += `Here's what you know about the user:\n\n`;

    // Add episodic memories
    if (episodic.length > 0) {
        prompt += `PAST EXPERIENCES:\n`;
        episodic.forEach((m, i) => {
            prompt += `${i + 1}. ${m.title}: ${m.content}\n`;
        });
        prompt += `\n`;
    }

    // Add entities
    if (entities.length > 0) {
        prompt += `PEOPLE & THINGS THEY KNOW:\n`;
        entities.forEach((m, i) => {
            prompt += `${i + 1}. ${m.title}: ${m.content}\n`;
        });
        prompt += `\n`;
    }

    // Add facts
    if (facts.length > 0) {
        prompt += `PREFERENCES & KNOWLEDGE:\n`;
        facts.forEach((m, i) => {
            prompt += `${i + 1}. ${m.title}: ${m.content}\n`;
        });
        prompt += `\n`;
    }

    prompt += `Use this information to provide relevant, personalized responses. Reference specific memories when appropriate.`;

    return prompt;
}

/**
 * Process OpenAI's streaming response
 * This reads the stream chunk by chunk and calls onChunk for each piece
 */
async function processStreamingResponse(response, onChunk) {
    const reader = response.body.getReader();
    const decoder = new TextDecoder('utf-8');
    let fullResponse = '';

    try {
        while (true) {
            const { done, value } = await reader.read();

            if (done) {
                break;
            }

            // Decode chunk
            const chunk = decoder.decode(value, { stream: true });
            const lines = chunk.split('\n');

            for (const line of lines) {
                if (line.startsWith('data: ')) {
                    const data = line.substring(6);

                    if (data === '[DONE]') {
                        continue;
                    }

                    try {
                        const parsed = JSON.parse(data);
                        const content = parsed.choices[0]?.delta?.content;

                        if (content) {
                            fullResponse += content;
                            if (onChunk) {
                                onChunk(content);
                            }
                        }
                    } catch (parseError) {
                        // Skip invalid JSON lines
                        continue;
                    }
                }
            }
        }
    } catch (error) {
        console.error('Streaming error:', error);
        throw error;
    }

    return fullResponse;
}

/**
 * Get chat history
 * @returns {Array} Array of messages
 */
export function getChatHistory() {
    return [...chatHistory];
}

/**
 * Clear chat history
 */
export function clearChatHistory() {
    chatHistory = [];
    console.log('✅ Chat history cleared');
}

/**
 * Get memory count used in current session
 * @param {number} userId
 * @returns {Promise<number>}
 */
export async function getMemoryContextCount(userId) {
    const memories = await getMemories(userId);
    return memories.length;
}

/**
 * Preview the system prompt that will be sent to the AI
 * Useful for students to see EXACTLY what context the AI receives
 *
 * @param {number} userId
 * @returns {Promise<string>} System prompt
 */
export async function previewSystemPrompt(userId) {
    const memories = await getMemories(userId);
    return buildSystemPromptWithMemories(memories);
}

/**
 * Estimate token count for current context
 * Rough estimate: 1 token ≈ 4 characters
 *
 * @param {number} userId
 * @returns {Promise<number>} Approximate token count
 */
export async function estimateTokenCount(userId) {
    const systemPrompt = await previewSystemPrompt(userId);
    const chatText = chatHistory.map(m => m.content).join(' ');
    const totalText = systemPrompt + chatText;

    // Rough estimate: 1 token per 4 characters
    return Math.ceil(totalText.length / 4);
}

console.log('💬 Chat module loaded');
