/**
 * chat-ui.js - Chat tab UI logic
 */

import { sendMessage, clearChatHistory, getChatHistory } from './chat.js';

let userId = null;
let apiKey = null;

export function init(currentUserId, currentApiKey) {
    userId = currentUserId;
    apiKey = currentApiKey;

    setupChatListeners();
    displayChatHistory();
}

function setupChatListeners() {
    const sendBtn = document.getElementById('send-btn');
    const messageInput = document.getElementById('message-input');
    const clearBtn = document.getElementById('clear-chat-btn');

    if (sendBtn) {
        sendBtn.addEventListener('click', handleSendMessage);
    }

    if (messageInput) {
        messageInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                handleSendMessage();
            }
        });
    }

    if (clearBtn) {
        clearBtn.addEventListener('click', () => {
            if (confirm('Clear chat history?')) {
                clearChatHistory();
                displayChatHistory();
            }
        });
    }
}

async function handleSendMessage() {
    if (!apiKey) {
        alert('Please add your OpenAI API key in Settings!');
        return;
    }

    const input = document.getElementById('message-input');
    const message = input.value.trim();

    if (!message) return;

    // Clear input
    input.value = '';

    // Add user message to UI
    addMessageToUI('user', message);

    // Show typing indicator
    const typingId = addTypingIndicator();

    try {
        let assistantMessage = '';

        await sendMessage(apiKey, userId, message, (chunk) => {
            assistantMessage += chunk;
            updateTypingIndicator(typingId, assistantMessage);
        });

        removeTypingIndicator(typingId);

    } catch (error) {
        removeTypingIndicator(typingId);
        addMessageToUI('assistant', `Error: ${error.message}`);
    }
}

function addMessageToUI(role, content) {
    const messagesContainer = document.getElementById('chat-messages');
    if (!messagesContainer) return;

    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${role}-message`;
    messageDiv.textContent = content;

    messagesContainer.appendChild(messageDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

function addTypingIndicator() {
    const id = Date.now();
    const messagesContainer = document.getElementById('chat-messages');
    if (!messagesContainer) return id;

    const typingDiv = document.createElement('div');
    typingDiv.className = 'message assistant-message typing';
    typingDiv.id = `typing-${id}`;
    typingDiv.innerHTML = '<span class="typing-dots">...</span>';

    messagesContainer.appendChild(typingDiv);
    messagesContainer.scrollTop = messagesContainer.scrollHeight;

    return id;
}

function updateTypingIndicator(id, content) {
    const typingDiv = document.getElementById(`typing-${id}`);
    if (typingDiv) {
        typingDiv.textContent = content;
        typingDiv.className = 'message assistant-message';
    }
}

function removeTypingIndicator(id) {
    const typingDiv = document.getElementById(`typing-${id}`);
    if (typingDiv) {
        typingDiv.remove();
    }
}

function displayChatHistory() {
    const messagesContainer = document.getElementById('chat-messages');
    if (!messagesContainer) return;

    messagesContainer.innerHTML = '';

    const history = getChatHistory();
    history.forEach(msg => {
        addMessageToUI(msg.role, msg.content);
    });
}
