/**
 * auth.js - Local authentication system
 *
 * This module handles user authentication WITHOUT a backend server.
 * Students will learn:
 * - How to securely hash passwords using Web Crypto API (built-in browser API)
 * - How to manage user sessions with localStorage
 * - How to implement login/register flows
 *
 * Key concept: All authentication happens client-side. In production, you'd
 * do this server-side, but for learning and simple apps, this approach works!
 */

import { createUser, getUserByEmail } from './storage.js';

// Session key for localStorage
const SESSION_KEY = 'contextai_session';

/**
 * Hash a password using SHA-256 (Web Crypto API)
 * Students will learn: This is secure password storage without libraries!
 *
 * @param {string} password - Plain text password
 * @returns {Promise<string>} Hexadecimal hash string
 */
export async function hashPassword(password) {
    // Convert password string to bytes
    const encoder = new TextEncoder();
    const data = encoder.encode(password);

    // Use browser's built-in crypto API to hash
    // SHA-256 produces a 256-bit (32-byte) hash
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);

    // Convert hash bytes to hexadecimal string
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    const hashHex = hashArray
        .map(byte => byte.toString(16).padStart(2, '0'))
        .join('');

    return hashHex;
}

/**
 * Register a new user
 *
 * @param {string} username
 * @param {string} email
 * @param {string} password
 * @returns {Promise<Object>} { success: boolean, message: string, user?: Object }
 */
export async function register(username, email, password) {
    try {
        // Validation
        if (!username || username.trim().length < 2) {
            return {
                success: false,
                message: 'Username must be at least 2 characters'
            };
        }

        if (!email || !email.includes('@')) {
            return {
                success: false,
                message: 'Please enter a valid email address'
            };
        }

        if (!password || password.length < 6) {
            return {
                success: false,
                message: 'Password must be at least 6 characters'
            };
        }

        // Check if email already exists
        const existingUser = await getUserByEmail(email);
        if (existingUser) {
            return {
                success: false,
                message: 'An account with this email already exists'
            };
        }

        // Hash the password (NEVER store plain text passwords!)
        const passwordHash = await hashPassword(password);

        // Create user in database
        const userId = await createUser({
            username: username.trim(),
            email: email.toLowerCase().trim(),
            passwordHash
        });

        // Create session (log the user in immediately)
        const user = {
            id: userId,
            username: username.trim(),
            email: email.toLowerCase().trim()
        };

        setSession(user);

        return {
            success: true,
            message: 'Account created successfully!',
            user
        };
    } catch (error) {
        console.error('Registration error:', error);
        return {
            success: false,
            message: 'Registration failed. Please try again.'
        };
    }
}

/**
 * Login an existing user
 *
 * @param {string} email
 * @param {string} password
 * @returns {Promise<Object>} { success: boolean, message: string, user?: Object }
 */
export async function login(email, password) {
    try {
        // Validation
        if (!email || !password) {
            return {
                success: false,
                message: 'Please enter both email and password'
            };
        }

        // Find user by email
        const user = await getUserByEmail(email.toLowerCase().trim());
        if (!user) {
            return {
                success: false,
                message: 'Invalid email or password'
            };
        }

        // Hash the provided password and compare with stored hash
        const passwordHash = await hashPassword(password);

        if (passwordHash !== user.passwordHash) {
            return {
                success: false,
                message: 'Invalid email or password'
            };
        }

        // Create session
        const sessionUser = {
            id: user.id,
            username: user.username,
            email: user.email
        };

        setSession(sessionUser);

        return {
            success: true,
            message: 'Login successful!',
            user: sessionUser
        };
    } catch (error) {
        console.error('Login error:', error);
        return {
            success: false,
            message: 'Login failed. Please try again.'
        };
    }
}

/**
 * Logout the current user
 */
export function logout() {
    localStorage.removeItem(SESSION_KEY);
    console.log('✅ User logged out');
}

/**
 * Get the current logged-in user from session
 *
 * @returns {Object|null} User object or null if not logged in
 */
export function getCurrentUser() {
    try {
        const sessionData = localStorage.getItem(SESSION_KEY);
        if (!sessionData) {
            return null;
        }

        const user = JSON.parse(sessionData);
        return user;
    } catch (error) {
        console.error('Error reading session:', error);
        return null;
    }
}

/**
 * Check if a user is currently logged in
 *
 * @returns {boolean}
 */
export function isLoggedIn() {
    return getCurrentUser() !== null;
}

/**
 * Save user session to localStorage
 * (Private helper function)
 *
 * @param {Object} user - { id, username, email }
 */
function setSession(user) {
    localStorage.setItem(SESSION_KEY, JSON.stringify(user));
    console.log('✅ Session created for:', user.email);
}

/**
 * Validate email format (helper)
 *
 * @param {string} email
 * @returns {boolean}
 */
export function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

/**
 * Get user's initials for avatar display
 *
 * @param {Object} user - User object
 * @returns {string} Two-letter initials (e.g., "JD" for John Doe)
 */
export function getUserInitials(user) {
    if (!user || !user.username) {
        return '??';
    }

    const parts = user.username.split(' ');
    if (parts.length >= 2) {
        return (parts[0][0] + parts[1][0]).toUpperCase();
    }

    return user.username.substring(0, 2).toUpperCase();
}

// Log module load
console.log('🔐 Authentication module loaded');
