/**
 * app.js - Main application controller
 *
 * This module orchestrates the entire application:
 * - Initialize database
 * - Check authentication state
 * - Handle screen routing (login vs dashboard)
 * - Connect UI events to business logic
 *
 * Students will learn:
 * - How to structure a single-page app without frameworks
 * - State management patterns in vanilla JavaScript
 * - Event handling and DOM manipulation
 */

import { initDatabase } from './storage.js';
import { register, login, logout, getCurrentUser, isLoggedIn } from './auth.js';
import * as MemoryUI from './memory-ui.js';
import * as ChatUI from './chat-ui.js';
import * as UploadUI from './upload-ui.js';
import * as SettingsUI from './settings-ui.js';

// Application state
const state = {
    currentUser: null,
    currentTab: 'upload',
    apiKey: null
};

/**
 * Initialize the application
 * Called when DOM is ready
 */
export async function init() {
    console.log('🚀 Initializing Context AI...');

    try {
        // Initialize IndexedDB
        await initDatabase();

        // Check if user is logged in
        state.currentUser = getCurrentUser();

        // Load API key from localStorage
        state.apiKey = localStorage.getItem('openai_api_key');

        // Setup UI based on auth state
        if (isLoggedIn()) {
            showDashboard();
        } else {
            showAuthScreen();
        }

        // Setup global event listeners
        setupEventListeners();

        console.log('✅ App initialized successfully');
    } catch (error) {
        console.error('❌ Initialization failed:', error);
        showError('Failed to initialize app. Please refresh the page.');
    }
}

/**
 * Show authentication screen (login/register)
 */
function showAuthScreen() {
    document.getElementById('auth-screen').style.display = 'flex';
    document.getElementById('dashboard-screen').style.display = 'none';
}

/**
 * Show dashboard screen
 */
function showDashboard() {
    document.getElementById('auth-screen').style.display = 'none';
    document.getElementById('dashboard-screen').style.display = 'block';

    // Update user info display
    updateUserInfoDisplay();

    // Load initial tab
    switchTab(state.currentTab);
}

/**
 * Update user info in header
 */
function updateUserInfoDisplay() {
    const userNameEl = document.getElementById('user-name');
    const userEmailEl = document.getElementById('user-email');

    if (state.currentUser) {
        if (userNameEl) userNameEl.textContent = state.currentUser.username;
        if (userEmailEl) userEmailEl.textContent = state.currentUser.email;
    }
}

/**
 * Switch between dashboard tabs
 */
export function switchTab(tabName) {
    state.currentTab = tabName;

    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });

    // Remove active class from all tab buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });

    // Show selected tab
    const selectedTab = document.getElementById(`${tabName}-tab`);
    if (selectedTab) {
        selectedTab.classList.add('active');
    }

    // Activate tab button
    const selectedBtn = document.querySelector(`[data-tab="${tabName}"]`);
    if (selectedBtn) {
        selectedBtn.classList.add('active');
    }

    // Initialize tab content
    initializeTab(tabName);
}

/**
 * Initialize tab when switched to
 */
async function initializeTab(tabName) {
    const userId = state.currentUser.id;

    switch (tabName) {
        case 'upload':
            UploadUI.init(userId, state.apiKey);
            break;
        case 'memories':
            await MemoryUI.init(userId);
            break;
        case 'chat':
            ChatUI.init(userId, state.apiKey);
            break;
        case 'settings':
            SettingsUI.init(state.apiKey);
            break;
    }
}

/**
 * Setup global event listeners
 */
function setupEventListeners() {
    // Auth screen listeners
    setupAuthListeners();

    // Dashboard listeners
    setupDashboardListeners();
}

/**
 * Setup authentication event listeners
 */
function setupAuthListeners() {
    // Toggle between login and register
    const showRegisterBtn = document.getElementById('show-register');
    const showLoginBtn = document.getElementById('show-login');

    if (showRegisterBtn) {
        showRegisterBtn.addEventListener('click', (e) => {
            e.preventDefault();
            toggleAuthForms('register');
        });
    }

    if (showLoginBtn) {
        showLoginBtn.addEventListener('click', (e) => {
            e.preventDefault();
            toggleAuthForms('login');
        });
    }

    // Login form submit
    const loginForm = document.getElementById('login-form');
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }

    // Register form submit
    const registerForm = document.getElementById('register-form');
    if (registerForm) {
        registerForm.addEventListener('submit', handleRegister);
    }
}

/**
 * Setup dashboard event listeners
 */
function setupDashboardListeners() {
    // Tab buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const tabName = btn.getAttribute('data-tab');
            switchTab(tabName);
        });
    });

    // Logout button
    const logoutBtn = document.getElementById('logout-btn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', handleLogout);
    }
}

/**
 * Toggle between login and register forms
 */
function toggleAuthForms(formType) {
    const loginContainer = document.getElementById('login-container');
    const registerContainer = document.getElementById('register-container');

    if (formType === 'register') {
        loginContainer.style.display = 'none';
        registerContainer.style.display = 'block';
    } else {
        loginContainer.style.display = 'block';
        registerContainer.style.display = 'none';
    }
}

/**
 * Handle login form submission
 */
async function handleLogin(e) {
    e.preventDefault();

    const email = document.getElementById('login-email').value;
    const password = document.getElementById('login-password').value;
    const errorEl = document.getElementById('login-error');

    try {
        showLoading('Logging in...');

        const result = await login(email, password);

        if (result.success) {
            state.currentUser = result.user;
            hideLoading();
            showDashboard();
        } else {
            hideLoading();
            errorEl.textContent = result.message;
            errorEl.style.display = 'block';
        }
    } catch (error) {
        hideLoading();
        errorEl.textContent = 'Login failed. Please try again.';
        errorEl.style.display = 'block';
    }
}

/**
 * Handle register form submission
 */
async function handleRegister(e) {
    e.preventDefault();

    const username = document.getElementById('register-username').value;
    const email = document.getElementById('register-email').value;
    const password = document.getElementById('register-password').value;
    const errorEl = document.getElementById('register-error');

    try {
        showLoading('Creating account...');

        const result = await register(username, email, password);

        if (result.success) {
            state.currentUser = result.user;
            hideLoading();
            showDashboard();
        } else {
            hideLoading();
            errorEl.textContent = result.message;
            errorEl.style.display = 'block';
        }
    } catch (error) {
        hideLoading();
        errorEl.textContent = 'Registration failed. Please try again.';
        errorEl.style.display = 'block';
    }
}

/**
 * Handle logout
 */
function handleLogout() {
    if (confirm('Are you sure you want to log out?')) {
        logout();
        state.currentUser = null;
        state.apiKey = null;
        showAuthScreen();
    }
}

/**
 * Show loading indicator
 */
function showLoading(message = 'Loading...') {
    // You can implement a loading overlay here
    console.log('Loading:', message);
}

/**
 * Hide loading indicator
 */
function hideLoading() {
    console.log('Loading complete');
}

/**
 * Show error message
 */
function showError(message) {
    alert(message); // Simple for now, can be enhanced with a toast notification
}

/**
 * Update API key in state and persist to localStorage
 */
export function updateApiKey(apiKey) {
    state.apiKey = apiKey;
    localStorage.setItem('openai_api_key', apiKey);
}

/**
 * Get current API key
 */
export function getApiKey() {
    return state.apiKey;
}

/**
 * Get current user
 */
export function getCurrentAppUser() {
    return state.currentUser;
}

// Initialize app when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}

console.log('📱 App controller loaded');
