# Context AI - Teaching Template

A simplified version of Context AI for learning how to build AI applications with **Memory-First Architecture**.

This is a complete, working template that demonstrates the core pattern behind ChatGPT and successful AI products: **persistent memory that makes AI personal and useful**.

---

## What You'll Learn

After going through this template, you'll understand:

- **Memory-First Architecture** - The pattern that separates AI toys from tools people use daily
- **Local-first development** - Building apps that work entirely in the browser
- **Secure authentication** - Password hashing without a backend server
- **OpenAI API integration** - Direct API calls with streaming responses
- **Prompt engineering** - Designing prompts that extract structured data
- **Context injection** - How memory transforms AI conversations
- **IndexedDB** - Browser-native database for structured data
- **State management** - Vanilla JavaScript patterns without frameworks

---

## Quick Start (5 minutes)

### 1. Get Your OpenAI API Key

Visit [platform.openai.com/api-keys](https://platform.openai.com/api-keys) and create a new API key. Copy and save it.

**Cost estimate:** $1-5 for the workshop (100 conversations + 50 chat messages)

### 2. Open the App

Simply open `index.html` in your browser. No installation, no build tools needed!

Or double-click `app.html` to skip the landing page and go straight to the app.

### 3. Create an Account

Your account is stored locally in your browser (IndexedDB). No server, no cloud storage.

### 4. Add Your API Key

Go to Settings → Paste your OpenAI API key → Click "Save"

### 5. Upload ChatGPT Conversations

**How to export from ChatGPT:**
1. Open ChatGPT
2. Click your profile → Settings
3. Data Controls → Export data
4. Download the ZIP file
5. Extract and find `conversations.json`

**Upload the file:**
- Drag and drop onto the upload zone, OR
- Click "Choose File" and select `conversations.json`

### 6. Watch the Magic

The app will:
1. Parse each conversation
2. Call OpenAI to extract memories (episodic, entities, facts)
3. Store memories in IndexedDB
4. Display them in the Memories tab

### 7. Chat with Memory Context

Go to the Chat tab and ask questions. The AI will use ALL your memories to provide personalized responses!

---

## How It Works

### The Memory-First Architecture Pattern

```
Conversation → Memory Extraction → Storage → Context Injection → Personalized AI
```

**Step 1: Memory Extraction**
```javascript
// In memory-extractor.js
const MEMORY_EXTRACTION_PROMPT = `
Analyze this conversation and extract:
1. Episodic memories (events, experiences)
2. Entities (people, places, organizations)
3. Facts (preferences, skills, knowledge)

Return JSON format.
`;
```

**Step 2: Storage**
```javascript
// In storage.js
await saveMemory({
  userId: 1,
  type: 'episodic',
  title: 'Built first AI app',
  content: 'Learned Memory-First Architecture on Oct 25'
});
```

**Step 3: Context Injection**
```javascript
// In chat.js
const systemPrompt = `
You are a personal AI assistant with access to the user's memories:

PAST EXPERIENCES:
- Built first AI app: Learned Memory-First Architecture on Oct 25

PREFERENCES & KNOWLEDGE:
- Prefers vanilla JavaScript over React for teaching
- Knows how to use IndexedDB for browser storage

Use these memories to provide personalized responses.
`;
```

**Step 4: Personalized Responses**
```
User: "What did I work on recently?"
AI: "You recently built your first AI app and learned about Memory-First
     Architecture on October 25th! You also learned how to use IndexedDB
     for browser storage. How did that project go?"
```

---

## File Structure

```
Lighting_Code_Template/
├── index.html                 # Landing page (marketing)
├── app.html                   # Main application
├── README.md                  # This file
├── css/
│   ├── landing.css            # Landing page styles
│   └── app.css                # Application styles (Apple-inspired)
└── js/
    ├── storage.js             # IndexedDB wrapper (CRUD operations)
    ├── auth.js                # Local authentication (password hashing)
    ├── memory-extractor.js    # OpenAI memory extraction (THE KEY MODULE)
    ├── chat.js                # Chat with memory context
    ├── app.js                 # Main controller
    ├── upload-ui.js           # Upload tab UI logic
    ├── memory-ui.js           # Memories tab UI logic
    ├── chat-ui.js             # Chat tab UI logic
    └── settings-ui.js         # Settings tab UI logic
```

---

## Code Walkthrough

### 1. storage.js - IndexedDB Wrapper

**What it does:** Provides CRUD operations for storing users, conversations, and memories locally in the browser.

**Key functions:**
- `initDatabase()` - Creates IndexedDB with 3 object stores
- `saveMemory()` - Stores a memory with type, title, content
- `getMemories(userId, type)` - Retrieves memories with optional filtering
- `searchMemories(keyword)` - Simple keyword search (students can upgrade to embeddings!)

**Learning points:**
- How IndexedDB works (NoSQL database in the browser)
- Creating indexes for fast queries
- Promise-based async operations

**Try modifying:**
- Add a new memory type (e.g., "goals", "hobbies")
- Implement memory editing/deletion
- Add a "favorite" flag to memories

---

### 2. auth.js - Local Authentication

**What it does:** Implements user registration and login WITHOUT a backend server.

**Key functions:**
- `hashPassword()` - Uses Web Crypto API (SHA-256) to hash passwords
- `register()` - Creates new user account
- `login()` - Verifies credentials
- `getCurrentUser()` - Gets logged-in user from localStorage

**Learning points:**
- Secure password hashing without libraries
- Session management with localStorage
- Client-side validation

**Security note:** In production, you'd do this server-side. But for learning and simple apps, client-side auth works!

**Try modifying:**
- Add password strength indicator
- Implement "remember me" functionality
- Add profile picture support

---

### 3. memory-extractor.js - THE CORE PATTERN

**What it does:** Uses OpenAI API to transform conversations into structured memories.

**The extraction prompt (students can see and modify):**
```javascript
const MEMORY_EXTRACTION_PROMPT = `
You are a memory extraction AI. Analyze a conversation and extract:

1. EPISODIC MEMORIES: Specific events/experiences
2. ENTITIES: People, places, organizations
3. FACTS: User preferences, skills, knowledge

Return JSON:
{ episodic: [...], entities: [...], facts: [...] }
`;
```

**Key functions:**
- `extractMemoriesFromConversation()` - Calls OpenAI with extraction prompt
- `processUploadedFile()` - Processes entire ChatGPT export
- `parseMemoryResponse()` - Extracts JSON from AI response

**Learning points:**
- How to design prompts that return consistent JSON
- Working with OpenAI API directly
- Error handling for API calls

**Try modifying:**
- Change the prompt to extract different information
- Add new memory categories (hobbies, goals, routines)
- Implement batch processing for better rate limits

---

### 4. chat.js - Chat with Memory Context

**What it does:** Injects ALL user memories into the system prompt for personalized AI conversations.

**The magic happens here:**
```javascript
async function sendMessage(apiKey, userId, userMessage) {
  // Get ALL memories
  const memories = await getMemories(userId);

  // Build system prompt with full context
  const systemPrompt = buildSystemPromptWithMemories(memories);

  // Call OpenAI with streaming
  const messages = [
    { role: 'system', content: systemPrompt },
    ...chatHistory
  ];

  // Stream response
  const response = await fetch(OPENAI_API_URL, {
    method: 'POST',
    body: JSON.stringify({ model: 'gpt-4o-mini', messages, stream: true })
  });
}
```

**Key functions:**
- `sendMessage()` - Sends message with memory context
- `buildSystemPromptWithMemories()` - Creates prompt with all memories
- `processStreamingResponse()` - Handles real-time AI responses

**Learning points:**
- How context transforms AI behavior
- OpenAI streaming API
- Managing token limits

**Try modifying:**
- Implement semantic search to limit memories (top 10 most relevant)
- Add memory citations (show which memories were referenced)
- Let users manually select which memories to include

---

### 5. app.js - Main Controller

**What it does:** Orchestrates the entire application - initialization, routing, state management.

**Key functions:**
- `init()` - Initialize database and check auth state
- `switchTab()` - Handle tab navigation
- `handleLogin()` / `handleRegister()` - Auth flows
- `showDashboard()` / `showAuthScreen()` - Screen routing

**Learning points:**
- State management without frameworks
- Event handling patterns
- Module orchestration

---

## Customization Ideas

### Easy (30 minutes)

✅ **Change the memory extraction prompt** (memory-extractor.js)
- Add new categories like "goals", "habits", "skills"
- Extract different information from conversations

✅ **Customize UI colors** (app.css, landing.css)
- Change accent color from blue to purple
- Add a light mode

✅ **Add memory statistics** (memory-ui.js)
- Show counts by type: 10 episodic, 5 entities, 8 facts
- Display total memory count in header

### Medium (1-2 hours)

✅ **Implement tag-based filtering**
- Extract tags during memory extraction
- Add tag filter UI
- Search by tags

✅ **Add memory editing/deletion**
- Click memory card to edit
- Delete button with confirmation
- Update IndexedDB

✅ **Create a timeline view**
- Sort memories by date
- Group by week/month
- Visual timeline UI

✅ **Support other conversation formats**
- Parse Slack exports
- Parse Discord exports
- Generic conversation JSON structure

### Advanced (4-8 hours)

✅ **Implement client-side semantic search**
- Use OpenAI embeddings API to create vectors
- Store vectors in IndexedDB
- Calculate cosine similarity for search
- Retrieve most relevant memories for chat context

✅ **Add conversation summarization**
- Summarize long conversations before extraction
- Show conversation summaries in Memories tab
- Link memories to source conversations

✅ **Build memory graphs**
- Visualize connections between memories
- Show entity relationships
- Interactive graph UI with D3.js

✅ **Multi-user support**
- User switching without logout
- User profiles
- Data export/import per user

---

## Cost Estimate

### OpenAI API Costs (as of 2024)

**gpt-4o-mini pricing:**
- Input: $0.150 per 1M tokens (~$0.0001 per message)
- Output: $0.600 per 1M tokens (~$0.0004 per response)

**Workshop estimate (100 conversations + 50 chat messages):**
- Memory extraction: 100 conversations × ~1000 tokens = 100K tokens = **$0.10**
- Chat: 50 messages × ~1500 tokens = 75K tokens = **$0.05**
- **Total: ~$0.15 - $5** (depending on conversation length)

**Tips to minimize costs:**
- Use shorter conversations for testing
- Cache extracted memories (don't re-extract)
- Limit chat history (only keep recent messages)

---

## Troubleshooting

### "API request failed"
- Check your API key in Settings
- Make sure you have credits in your OpenAI account
- Test your API key using the "Test" button

### "No memories found"
- Upload conversations first
- Check browser console for errors during extraction
- Make sure OpenAI returned valid JSON

### "Database initialization failed"
- Try a different browser (Chrome/Edge recommended)
- Clear browser data and refresh
- Check that IndexedDB is enabled

### Chat not working
- Add API key in Settings
- Upload conversations first (need memories for context)
- Check browser console for errors

### Slow memory extraction
- Normal! Each conversation takes 2-5 seconds
- 100 conversations = 3-8 minutes
- Avoid uploading 1000+ conversations at once

---

## Learning Resources

### Official Documentation

**OpenAI API**
- [API Reference](https://platform.openai.com/docs/api-reference)
- [Chat Completions](https://platform.openai.com/docs/guides/chat-completions)
- [Streaming](https://platform.openai.com/docs/api-reference/streaming)

**IndexedDB**
- [MDN Guide](https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API)
- [Using IndexedDB](https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API/Using_IndexedDB)

**Web Crypto API**
- [MDN Documentation](https://developer.mozilla.org/en-US/docs/Web/API/Web_Crypto_API)
- [SubtleCrypto](https://developer.mozilla.org/en-US/docs/Web/API/SubtleCrypto)

### Recommended Reading

- [Prompt Engineering Guide](https://www.promptingguide.ai/)
- [OpenAI Best Practices](https://platform.openai.com/docs/guides/prompt-engineering)
- [Memory in AI Systems](https://github.com/langchain-ai/langchain/discussions/5815)

---

## What's Next?

### After mastering this template:

1. **Deploy your own version**
   - Host on GitHub Pages (it's just static files!)
   - Share with friends
   - Build your own personal AI memory database

2. **Add advanced features**
   - Implement vector search with embeddings
   - Add support for documents (PDFs, notes)
   - Build memory graphs and visualizations

3. **Build your own AI app**
   - Use this template as a starting point
   - Apply Memory-First Architecture to your idea
   - Ship a product people use every day!

### Ideas for AI apps using this pattern:

- **Personal journal AI** - Remembers all your thoughts and reflections
- **Meeting notes AI** - Recalls every meeting and action item
- **Learning companion** - Tracks what you've learned and helps you review
- **Project memory** - Remembers context across long-term projects
- **Family historian** - Stores family stories and memories

---

## FAQs

**Q: Is this production-ready?**
A: No, it's a teaching template. For production:
- Move authentication to a backend server
- Use a real database (not just IndexedDB)
- Implement proper security measures
- Add rate limiting and error handling
- Use vector databases for semantic search

**Q: Can I use this for my own projects?**
A: Absolutely! That's the point. Modify, extend, deploy.

**Q: Why vanilla JavaScript instead of React?**
A: Simplicity. You can understand every line of code in 30 minutes. No build tools, no abstractions.

**Q: How is this different from ChatGPT's memory feature?**
A: ChatGPT's memory:
- Locked into ChatGPT only
- Opaque (you can't see the extraction logic)
- Limited to OpenAI's ecosystem

Context AI pattern:
- Portable memory you can use with any AI
- Transparent prompts you can modify
- Own your data completely

**Q: What if I want to use Claude instead of OpenAI?**
A: Easy! Modify `memory-extractor.js` and `chat.js` to call Claude's API instead. Same pattern works.

---

## Contributing

Found a bug? Have an idea? This is a learning template - improvements welcome!

**Suggested improvements:**
- Add more example conversations
- Improve error messages
- Add dark/light mode toggle
- Create video walkthrough
- Write blog post tutorial

---

## Credits

Built for **AI Builder Workshop** by Nathan Wang.

Based on the Memory-First Architecture pattern used in successful AI products like ChatGPT, Context AI, and Notion AI.

**License:** MIT - Use however you want, just keep learning!

---

## Contact

Questions? Ideas? Want to show what you built?

- Twitter: [@AI_Nate_SA](https://twitter.com/AI_Nate_SA)
- Discord: [AI Builder Community](https://discord.com/invite/rZ8Uu925DY)
- GitHub: Share your modifications!

---

**Remember:** 90% of AI demos fail because they lack memory. Build apps with Memory-First Architecture and create tools people use every day.

Happy building! 🚀
